<?php
/**
 * Plugin Name: LLMs.txt Generator
 * Description: Nástroj pro generování a správu souboru llms.txt pro AI agenty s logováním přístupů.
 * Version: 1.7
 * Author: Michal Kubíček
 * Author URI: https://optimalizace.ai/llms
 * License: GPL2
 * Text Domain: llms-txt-generator
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Zamezení přímému přístupu
}

/**
 * Hlavní třída pluginu
 */
class OKLLMS_Generator {

	private $option_key = 'okllms_settings';
	private $table_name;
	private $api_endpoint = 'https://optimalizace.ai/api/llms-stats/ping.php';
	private $plugin_version = '1.7';

	/**
	 * Známí AI roboti a jejich identifikace
	 */
	private $known_bots = array(
		'GPTBot'           => array( 'name' => 'OpenAI GPTBot', 'company' => 'OpenAI' ),
		'ChatGPT-User'     => array( 'name' => 'ChatGPT User', 'company' => 'OpenAI' ),
		'OAI-SearchBot'    => array( 'name' => 'OpenAI SearchBot', 'company' => 'OpenAI' ),
		'ClaudeBot'        => array( 'name' => 'Claude Bot', 'company' => 'Anthropic' ),
		'Claude-Web'       => array( 'name' => 'Claude Web', 'company' => 'Anthropic' ),
		'anthropic-ai'     => array( 'name' => 'Anthropic AI', 'company' => 'Anthropic' ),
		'Google-Extended'  => array( 'name' => 'Google AI (Bard/Gemini)', 'company' => 'Google' ),
		'Googlebot'        => array( 'name' => 'Googlebot', 'company' => 'Google' ),
		'Bingbot'          => array( 'name' => 'Bingbot', 'company' => 'Microsoft' ),
		'bingbot'          => array( 'name' => 'Bingbot', 'company' => 'Microsoft' ),
		'CCBot'            => array( 'name' => 'Common Crawl Bot', 'company' => 'Common Crawl' ),
		'Bytespider'       => array( 'name' => 'Bytespider', 'company' => 'ByteDance' ),
		'PerplexityBot'    => array( 'name' => 'Perplexity Bot', 'company' => 'Perplexity' ),
		'Applebot'         => array( 'name' => 'Applebot', 'company' => 'Apple' ),
		'Applebot-Extended' => array( 'name' => 'Applebot Extended', 'company' => 'Apple' ),
		'cohere-ai'        => array( 'name' => 'Cohere AI', 'company' => 'Cohere' ),
		'YouBot'           => array( 'name' => 'You.com Bot', 'company' => 'You.com' ),
		'Amazonbot'        => array( 'name' => 'Amazonbot', 'company' => 'Amazon' ),
		'FacebookBot'      => array( 'name' => 'Facebook Bot', 'company' => 'Meta' ),
		'Meta-ExternalAgent' => array( 'name' => 'Meta External Agent', 'company' => 'Meta' ),
		'meta-externalagent' => array( 'name' => 'Meta External Agent', 'company' => 'Meta' ),
		'DuckDuckBot'      => array( 'name' => 'DuckDuckGo Bot', 'company' => 'DuckDuckGo' ),
		'YandexBot'        => array( 'name' => 'Yandex Bot', 'company' => 'Yandex' ),
		'Slurp'            => array( 'name' => 'Yahoo Slurp', 'company' => 'Yahoo' ),
		'SemrushBot'       => array( 'name' => 'Semrush Bot', 'company' => 'Semrush' ),
		'AhrefsBot'        => array( 'name' => 'Ahrefs Bot', 'company' => 'Ahrefs' ),
		'MJ12bot'          => array( 'name' => 'Majestic Bot', 'company' => 'Majestic' ),
		'DotBot'           => array( 'name' => 'DotBot', 'company' => 'Moz' ),
		'PetalBot'         => array( 'name' => 'Petal Bot', 'company' => 'Huawei' ),
		'Scrapy'           => array( 'name' => 'Scrapy', 'company' => 'Scraper' ),
		'curl'             => array( 'name' => 'cURL', 'company' => 'CLI Tool' ),
		'Wget'             => array( 'name' => 'Wget', 'company' => 'CLI Tool' ),
		'python-requests'  => array( 'name' => 'Python Requests', 'company' => 'Script' ),
	);

	public function __construct() {
		global $wpdb;
		$this->table_name = $wpdb->prefix . 'okllms_access_log';

		// Hook pro nastavení pluginu
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		
		// Hook pro zachycení URL /llms.txt
		add_action( 'template_redirect', array( $this, 'catch_llms_request' ) );
		
		// Přidání odkazů na stránku pluginů
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'add_plugin_links' ) );
		
		// Discovery link v hlavičce
		add_action( 'wp_head', array( $this, 'add_head_discovery' ) );
		
		// Filtr pro robots.txt
		add_filter( 'robots_txt', array( $this, 'add_robots_rule' ), 10, 2 );

		// Aktivace pluginu - vytvoření tabulky
		register_activation_hook( __FILE__, array( $this, 'activate_plugin' ) );
	}

	/**
	 * Přidání pravidla do robots.txt pro povolení přístupu k llms.txt
	 */
	public function add_robots_rule( $output, $public ) {
		// Přidáme pravidlo pouze pokud web není skrytý před vyhledávači
		if ( $public ) {
			// Kontrola, zda pravidlo již není přítomno
			if ( strpos( $output, 'Allow: /llms.txt' ) === false ) {
				$output .= "\n# LLMs.txt - informace pro AI agenty\n";
				$output .= "Allow: /llms.txt\n";
			}
		}
		return $output;
	}

	/**
	 * Získání hash webu pro anonymizaci
	 */
	private function get_site_hash() {
		return hash( 'sha256', home_url() );
	}

	/**
	 * Aktivace pluginu - vytvoření databázové tabulky
	 */
	public function activate_plugin() {
		global $wpdb;
		
		$charset_collate = $wpdb->get_charset_collate();
		
		$sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			accessed_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
			ip_address varchar(45) NOT NULL,
			user_agent text NOT NULL,
			bot_name varchar(100) DEFAULT NULL,
			bot_company varchar(100) DEFAULT NULL,
			is_bot tinyint(1) NOT NULL DEFAULT 0,
			referer text DEFAULT NULL,
			PRIMARY KEY (id),
			KEY accessed_at (accessed_at),
			KEY bot_name (bot_name),
			KEY is_bot (is_bot)
		) $charset_collate;";
		
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );
		
		flush_rewrite_rules();
	}

	/**
	 * Přidání položek do menu
	 */
	public function add_admin_menu() {
		// Hlavní stránka nastavení
		add_options_page(
			'LLMs.txt Nastavení',
			'LLMs.txt',
			'manage_options',
			'llms-txt-generator',
			array( $this, 'render_admin_page' )
		);
	}

	/**
	 * Registrace nastavení pomocí WP Settings API
	 */
	public function register_settings() {
		register_setting( 'okllms_settings_group', $this->option_key );
	}

	/**
	 * Přidání odkazů do přehledu pluginů
	 */
	public function add_plugin_links( $links ) {
		$custom_links = array(
			'<a href="' . admin_url( 'options-general.php?page=llms-txt-generator' ) . '">Nastavení</a>',
			'<a href="' . admin_url( 'options-general.php?page=llms-txt-generator&tab=stats' ) . '">Statistiky</a>',
			'<a href="https://optimalizace.ai/llms" target="_blank" style="font-weight:bold;">Plný generátor</a>',
		);
		return array_merge( $custom_links, $links );
	}

	/**
	 * Discovery link pro AI boty
	 */
	public function add_head_discovery() {
		echo '<link rel="help" type="text/plain" href="' . esc_url( home_url( '/llms.txt' ) ) . '">' . PHP_EOL;
	}

	/**
	 * Identifikace robota z User-Agent
	 */
	private function identify_bot( $user_agent ) {
		foreach ( $this->known_bots as $identifier => $info ) {
			if ( stripos( $user_agent, $identifier ) !== false ) {
				return array(
					'is_bot'      => true,
					'bot_name'    => $info['name'],
					'bot_company' => $info['company'],
				);
			}
		}
		
		// Obecná detekce botů
		$bot_keywords = array( 'bot', 'crawler', 'spider', 'scraper', 'agent' );
		foreach ( $bot_keywords as $keyword ) {
			if ( stripos( $user_agent, $keyword ) !== false ) {
				return array(
					'is_bot'      => true,
					'bot_name'    => 'Neznámý bot',
					'bot_company' => 'Neznámá',
				);
			}
		}
		
		return array(
			'is_bot'      => false,
			'bot_name'    => null,
			'bot_company' => null,
		);
	}

	/**
	 * Odeslání statistiky na centrální server (non-blocking)
	 */
	private function send_to_central_server( $bot_info, $user_agent ) {
		$options = get_option( $this->option_key, array() );
		
		// Opt-out kontrola - výchozí je ZAPNUTO (pokud není explicitně vypnuto)
		if ( isset( $options['disable_stats'] ) && $options['disable_stats'] ) {
			return;
		}
		
		$data = array(
			'site_hash'      => $this->get_site_hash(),
			'site_domain'    => wp_parse_url( home_url(), PHP_URL_HOST ), // Vždy odesíláme doménu
			'site_url'       => home_url(),
			'site_name'      => get_bloginfo( 'name' ),
			'bot_name'       => $bot_info['bot_name'],
			'bot_company'    => $bot_info['bot_company'],
			'is_bot'         => $bot_info['is_bot'] ? 1 : 0,
			'user_agent'     => substr( $user_agent, 0, 1000 ),
			'plugin_version' => $this->plugin_version,
			'wp_version'     => get_bloginfo( 'version' ),
			'php_version'    => PHP_VERSION,
		);
		
		// Non-blocking request - neblokuje načítání llms.txt
		wp_remote_post( $this->api_endpoint, array(
			'body'      => wp_json_encode( $data ),
			'headers'   => array( 'Content-Type' => 'application/json' ),
			'timeout'   => 1,
			'blocking'  => false,
			'sslverify' => true,
		) );
	}

	/**
	 * Logování přístupu
	 */
	private function log_access() {
		global $wpdb;
		
		$user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( $_SERVER['HTTP_USER_AGENT'] ) : '';
		$ip_address = $this->get_client_ip();
		$referer    = isset( $_SERVER['HTTP_REFERER'] ) ? esc_url_raw( $_SERVER['HTTP_REFERER'] ) : '';
		
		$bot_info = $this->identify_bot( $user_agent );
		
		// Lokální logování
		$wpdb->insert(
			$this->table_name,
			array(
				'accessed_at'  => current_time( 'mysql' ),
				'ip_address'   => $ip_address,
				'user_agent'   => $user_agent,
				'bot_name'     => $bot_info['bot_name'],
				'bot_company'  => $bot_info['bot_company'],
				'is_bot'       => $bot_info['is_bot'] ? 1 : 0,
				'referer'      => $referer,
			),
			array( '%s', '%s', '%s', '%s', '%s', '%d', '%s' )
		);
		
		// Odeslání na centrální server
		$this->send_to_central_server( $bot_info, $user_agent );
	}

	/**
	 * Získání IP adresy klienta
	 */
	private function get_client_ip() {
		$ip_keys = array( 'HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR' );
		
		foreach ( $ip_keys as $key ) {
			if ( ! empty( $_SERVER[ $key ] ) ) {
				$ip = sanitize_text_field( $_SERVER[ $key ] );
				// Může obsahovat více IP (X-Forwarded-For), bereme první
				if ( strpos( $ip, ',' ) !== false ) {
					$ip = trim( explode( ',', $ip )[0] );
				}
				if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
					return $ip;
				}
			}
		}
		
		return '0.0.0.0';
	}

	/**
	 * Zachycení požadavku na /llms.txt
	 */
	public function catch_llms_request() {
		$request = trim( $_SERVER['REQUEST_URI'], '/' );
		
		// Odstranění query stringu
		if ( strpos( $request, '?' ) !== false ) {
			$request = substr( $request, 0, strpos( $request, '?' ) );
		}
		
		if ( $request === 'llms.txt' ) {
			// Logování přístupu
			$this->log_access();
			
			$options = get_option( $this->option_key );
			
			if ( ! $options ) {
				status_header( 404 );
				die( 'Soubor llms.txt neni nakonfigurovan.' );
			}

			// Rozlišení režimu
			if ( isset( $options['mode'] ) && 'manual' === $options['mode'] ) {
				$content = isset( $options['manual_text'] ) ? $options['manual_text'] : '';
			} else {
				$content = $this->generate_from_builder( $options );
			}

			header( 'Content-Type: text/plain; charset=utf-8' );
			header( 'Cache-Control: public, max-age=3600' );
			
			// UTF-8 BOM
			echo "\xEF\xBB\xBF" . $content;
			exit;
		}
	}

	/**
	 * Generování obsahu z Builderu
	 */
	private function generate_from_builder( $options ) {
		$name    = ! empty( $options['site_name'] ) ? $options['site_name'] : get_bloginfo( 'name' );
		$tagline = ! empty( $options['tagline'] ) ? $options['tagline'] : get_bloginfo( 'description' );
		$about   = ! empty( $options['about'] ) ? $options['about'] : '';

		$lines = array();
		$lines[] = '# ' . $name;
		$lines[] = '';
		$lines[] = '> ' . $tagline;
		
		if ( $about ) {
			$lines[] = '';
			$lines[] = '## O nas';
			$lines[] = '';
			$lines[] = trim( $about );
		}

		$lines[] = '';
		$lines[] = '## Sitemaps';
		$lines[] = '';
		$lines[] = '- [XML Sitemap](' . home_url( '/sitemap_index.xml' ) . ')';

		return implode( "\n", $lines );
	}

	/**
	 * Získání statistik
	 */
	private function get_stats( $days = 30 ) {
		global $wpdb;
		
		$date_from = date( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) );
		
		// Celkový počet přístupů
		$total = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) FROM {$this->table_name} WHERE accessed_at >= %s",
			$date_from
		) );
		
		// Počet přístupů od botů
		$bots = $wpdb->get_var( $wpdb->prepare(
			"SELECT COUNT(*) FROM {$this->table_name} WHERE accessed_at >= %s AND is_bot = 1",
			$date_from
		) );
		
		// Top boti
		$top_bots = $wpdb->get_results( $wpdb->prepare(
			"SELECT bot_name, bot_company, COUNT(*) as count 
			 FROM {$this->table_name} 
			 WHERE accessed_at >= %s AND is_bot = 1 AND bot_name IS NOT NULL
			 GROUP BY bot_name, bot_company 
			 ORDER BY count DESC 
			 LIMIT 20",
			$date_from
		) );
		
		// Přístupy po dnech
		$daily = $wpdb->get_results( $wpdb->prepare(
			"SELECT DATE(accessed_at) as date, 
			        SUM(CASE WHEN is_bot = 1 THEN 1 ELSE 0 END) as bots,
			        SUM(CASE WHEN is_bot = 0 THEN 1 ELSE 0 END) as humans
			 FROM {$this->table_name} 
			 WHERE accessed_at >= %s
			 GROUP BY DATE(accessed_at) 
			 ORDER BY date DESC",
			$date_from
		) );
		
		// Poslední přístupy
		$recent = $wpdb->get_results( $wpdb->prepare(
			"SELECT * FROM {$this->table_name} 
			 WHERE accessed_at >= %s
			 ORDER BY accessed_at DESC 
			 LIMIT 50",
			$date_from
		) );
		
		return array(
			'total'    => (int) $total,
			'bots'     => (int) $bots,
			'humans'   => (int) $total - (int) $bots,
			'top_bots' => $top_bots,
			'daily'    => $daily,
			'recent'   => $recent,
		);
	}

	/**
	 * Administrační rozhraní
	 */
	public function render_admin_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'settings';
		
		// Mazání logů
		if ( isset( $_POST['okllms_clear_logs'] ) && check_admin_referer( 'okllms_clear_logs_nonce' ) ) {
			global $wpdb;
			$wpdb->query( "TRUNCATE TABLE {$this->table_name}" );
			echo '<div class="updated"><p>Logy byly vymazány.</p></div>';
		}
		
		// Uložení dat
		if ( isset( $_POST['okllms_save'] ) ) {
			check_admin_referer( 'okllms_action_nonce' );
			
			$settings = array(
				'mode'          => sanitize_text_field( $_POST['okllms_mode'] ),
				'site_name'     => sanitize_text_field( $_POST['okllms_site_name'] ),
				'tagline'       => sanitize_text_field( $_POST['okllms_tagline'] ),
				'about'         => sanitize_textarea_field( $_POST['okllms_about'] ),
				'manual_text'   => wp_kses_post( $_POST['okllms_manual_text'] ),
				'disable_stats' => isset( $_POST['okllms_disable_stats'] ) ? 1 : 0,
			);
			
			update_option( $this->option_key, $settings );
			echo '<div class="updated"><p>Nastavení uloženo. <a href="' . esc_url( home_url( '/llms.txt' ) ) . '" target="_blank">Zobrazit llms.txt</a></p></div>';
		}

		$options = get_option( $this->option_key, array( 'mode' => 'builder' ) );
		$mode    = isset( $options['mode'] ) ? $options['mode'] : 'builder';
		?>
		<div class="wrap">
			<h1>LLMs.txt Generator</h1>
			
			<h2 class="nav-tab-wrapper">
				<a href="?page=llms-txt-generator&tab=settings" class="nav-tab <?php echo 'settings' === $current_tab ? 'nav-tab-active' : ''; ?>">
					<span class="dashicons dashicons-admin-settings" style="font-size: 16px; margin-top: 4px;"></span> Nastavení
				</a>
				<a href="?page=llms-txt-generator&tab=stats" class="nav-tab <?php echo 'stats' === $current_tab ? 'nav-tab-active' : ''; ?>">
					<span class="dashicons dashicons-chart-bar" style="font-size: 16px; margin-top: 4px;"></span> Statistiky přístupů
				</a>
				<a href="?page=llms-txt-generator&tab=logs" class="nav-tab <?php echo 'logs' === $current_tab ? 'nav-tab-active' : ''; ?>">
					<span class="dashicons dashicons-list-view" style="font-size: 16px; margin-top: 4px;"></span> Log přístupů
				</a>
			</h2>

			<?php if ( 'settings' === $current_tab ) : ?>
				<?php $this->render_settings_tab( $options, $mode ); ?>
			<?php elseif ( 'stats' === $current_tab ) : ?>
				<?php $this->render_stats_tab(); ?>
			<?php elseif ( 'logs' === $current_tab ) : ?>
				<?php $this->render_logs_tab(); ?>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Tab: Nastavení
	 */
	private function render_settings_tab( $options, $mode ) {
		?>
		<div class="notice notice-info" style="margin: 20px 0; padding: 15px; border-left-color: #2271b1;">
			<h2 style="margin: 0 0 10px 0;">
				<span class="dashicons dashicons-performance" style="vertical-align: middle;"></span> 
				Profesionální LLMs.txt Generator
			</h2>
			<p>Pro vytvoření pokročilého souboru včetně FAQ, ceníku a dalších sekcí využijte náš externí nástroj.</p>
			<p>
				<a href="https://optimalizace.ai/llms" target="_blank" class="button button-primary">
					Přejít na Optimalizace.ai/llms
				</a>
			</p>
			<p style="font-size: 12px; color: #646970;">
				Autor: Michal Kubíček | <a href="https://optimalizace.ai" target="_blank">optimalizace.ai</a>
			</p>
		</div>

		<form method="post" action="">
			<?php wp_nonce_field( 'okllms_action_nonce' ); ?>
			<input type="hidden" name="okllms_mode" id="okllms_mode" value="<?php echo esc_attr( $mode ); ?>">

			<h3 class="nav-tab-wrapper" style="border-bottom: 1px solid #ccc; margin-bottom: 20px;">
				<a href="#" class="nav-tab <?php echo 'builder' === $mode ? 'nav-tab-active' : ''; ?>" data-mode="builder">
					<span class="dashicons dashicons-edit" style="font-size: 17px; margin-top: 3px;"></span> Základní Builder
				</a>
				<a href="#" class="nav-tab <?php echo 'manual' === $mode ? 'nav-tab-active' : ''; ?>" data-mode="manual">
					<span class="dashicons dashicons-editor-code" style="font-size: 17px; margin-top: 3px;"></span> Vložit vlastní text
				</a>
			</h3>

			<div id="section-builder" class="tab-content" style="<?php echo 'builder' === $mode ? '' : 'display:none;'; ?>">
				<table class="form-table">
					<tr>
						<th scope="row"><label for="okllms_site_name">Název webu</label></th>
						<td><input type="text" name="okllms_site_name" id="okllms_site_name" value="<?php echo esc_attr( isset( $options['site_name'] ) ? $options['site_name'] : '' ); ?>" class="regular-text" placeholder="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>"></td>
					</tr>
					<tr>
						<th scope="row"><label for="okllms_tagline">Motto (Tagline)</label></th>
						<td><input type="text" name="okllms_tagline" id="okllms_tagline" value="<?php echo esc_attr( isset( $options['tagline'] ) ? $options['tagline'] : '' ); ?>" class="large-text" placeholder="<?php echo esc_attr( get_bloginfo( 'description' ) ); ?>"></td>
					</tr>
					<tr>
						<th scope="row"><label for="okllms_about">Stručně o nás</label></th>
						<td><textarea name="okllms_about" id="okllms_about" rows="5" class="large-text"><?php echo esc_textarea( isset( $options['about'] ) ? $options['about'] : '' ); ?></textarea></td>
					</tr>
				</table>
			</div>

			<div id="section-manual" class="tab-content" style="<?php echo 'manual' === $mode ? '' : 'display:none;'; ?>">
				<p>Zde vložte hotový Markdown kód vygenerovaný na webu <strong>optimalizace.ai/llms</strong>.</p>
				<textarea name="okllms_manual_text" rows="15" class="large-text" style="font-family: monospace; background: #23282d; color: #c3c4c7; padding: 15px;"><?php echo esc_textarea( isset( $options['manual_text'] ) ? $options['manual_text'] : '' ); ?></textarea>
			</div>

			<!-- Info o statistikách -->
			<div style="background: #f0f6fc; border: 1px solid #c3c4c7; border-radius: 4px; padding: 15px; margin: 20px 0;">
				<h3 style="margin-top: 0;">
					<span class="dashicons dashicons-chart-area" style="vertical-align: middle;"></span>
					Anonymní statistiky přístupů AI botů
				</h3>
				<p style="color: #646970; margin-bottom: 10px;">
					Plugin odesílá anonymní statistiky o přístupech AI botů k vašemu llms.txt. 
					Díky tomu vzniká unikátní přehled o tom, jak AI agenti (ChatGPT, Claude, Perplexity aj.) 
					přistupují k českým webům.
				</p>
				<p style="color: #646970; margin-bottom: 15px;">
					<strong>Co se odesílá:</strong> doména webu, název bota, user-agent, verze pluginu<br>
					<strong>Co se NEodesílá:</strong> IP adresy návštěvníků, obsah llms.txt, citlivá data
				</p>
				
				<table class="form-table" style="margin: 0 0 15px 0;">
					<tr>
						<th scope="row" style="padding: 10px 0;">Odesílání statistik</th>
						<td style="padding: 10px 0;">
							<label>
								<input type="checkbox" name="okllms_disable_stats" value="1" <?php checked( ! empty( $options['disable_stats'] ) ); ?> id="okllms_disable_stats">
								<strong>Vypnout</strong> odesílání anonymních statistik
							</label>
						</td>
					</tr>
				</table>
				
				<div id="stats-warning" style="background: #fff8e5; border: 1px solid #ffcc00; border-radius: 4px; padding: 12px; margin-bottom: 15px; <?php echo empty( $options['disable_stats'] ) ? 'display: none;' : ''; ?>">
					<p style="margin: 0; color: #6d5a00;">
						<span class="dashicons dashicons-warning" style="color: #d69e00;"></span>
						<strong>Děkujeme za zvážení!</strong> Anonymní data nám pomáhají vytvářet unikátní přehled o tom, 
						jak AI boti přistupují k českým webům. Pokud to zvážíte, budeme rádi, když odesílání ponecháte zapnuté. 
						Vaše data jsou v bezpečí a nikdy nesdílíme citlivé informace.
					</p>
				</div>
				
				<p style="margin-bottom: 0;">
					<a href="https://optimalizace.ai/api/llms-stats/stats.php" target="_blank" class="button button-primary">
						<span class="dashicons dashicons-chart-bar" style="font-size: 16px; margin-top: 4px;"></span>
						Zobrazit globální statistiky
					</a>
				</p>
			</div>
			
			<script>
			jQuery(document).ready(function($) {
				$('#okllms_disable_stats').on('change', function() {
					if ($(this).is(':checked')) {
						$('#stats-warning').slideDown(200);
					} else {
						$('#stats-warning').slideUp(200);
					}
				});
			});
			</script>

			<p class="submit">
				<input type="submit" name="okllms_save" class="button button-primary" value="Uložit a publikovat llms.txt">
				<a href="<?php echo esc_url( home_url( '/llms.txt' ) ); ?>" target="_blank" class="button" style="margin-left: 10px;">
					<span class="dashicons dashicons-external" style="font-size: 16px; margin-top: 4px;"></span> Náhled souboru
				</a>
			</p>
		</form>

		<script>
		jQuery(document).ready(function($) {
			$('.nav-tab[data-mode]').on('click', function(e) {
				e.preventDefault();
				var mode = $(this).data('mode');
				
				$('.nav-tab[data-mode]').removeClass('nav-tab-active');
				$(this).addClass('nav-tab-active');
				
				$('#okllms_mode').val(mode);
				$('.tab-content').hide();
				$('#section-' + mode).show();
			});
		});
		</script>
		<?php
	}

	/**
	 * Tab: Statistiky
	 */
	private function render_stats_tab() {
		$days  = isset( $_GET['days'] ) ? (int) $_GET['days'] : 30;
		$stats = $this->get_stats( $days );
		?>
		<div style="margin: 20px 0;">
			<form method="get" style="display: inline;">
				<input type="hidden" name="page" value="llms-txt-generator">
				<input type="hidden" name="tab" value="stats">
				<label for="days">Období: </label>
				<select name="days" id="days" onchange="this.form.submit()">
					<option value="7" <?php selected( $days, 7 ); ?>>Posledních 7 dní</option>
					<option value="30" <?php selected( $days, 30 ); ?>>Posledních 30 dní</option>
					<option value="90" <?php selected( $days, 90 ); ?>>Posledních 90 dní</option>
					<option value="365" <?php selected( $days, 365 ); ?>>Poslední rok</option>
				</select>
			</form>
		</div>

		<!-- Přehled -->
		<div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px;">
			<div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); border-left: 4px solid #2271b1;">
				<div style="font-size: 32px; font-weight: 700; color: #2271b1;"><?php echo number_format( $stats['total'] ); ?></div>
				<div style="color: #646970;">Celkem přístupů</div>
			</div>
			<div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); border-left: 4px solid #00a32a;">
				<div style="font-size: 32px; font-weight: 700; color: #00a32a;"><?php echo number_format( $stats['bots'] ); ?></div>
				<div style="color: #646970;">Přístupy od botů</div>
			</div>
			<div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); border-left: 4px solid #dba617;">
				<div style="font-size: 32px; font-weight: 700; color: #dba617;"><?php echo number_format( $stats['humans'] ); ?></div>
				<div style="color: #646970;">Ostatní přístupy</div>
			</div>
			<div style="background: #fff; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); border-left: 4px solid #8c5dd5;">
				<div style="font-size: 32px; font-weight: 700; color: #8c5dd5;"><?php echo count( $stats['top_bots'] ); ?></div>
				<div style="color: #646970;">Různých botů</div>
			</div>
		</div>

		<!-- Top boti -->
		<h3>Top AI boti a crawlery</h3>
		<?php if ( ! empty( $stats['top_bots'] ) ) : ?>
		<table class="wp-list-table widefat fixed striped" style="margin-bottom: 30px;">
			<thead>
				<tr>
					<th style="width: 50px;">#</th>
					<th>Bot</th>
					<th>Společnost</th>
					<th style="width: 120px;">Počet přístupů</th>
					<th style="width: 100px;">Podíl</th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $stats['top_bots'] as $i => $bot ) : ?>
				<tr>
					<td><?php echo $i + 1; ?></td>
					<td>
						<strong><?php echo esc_html( $bot->bot_name ); ?></strong>
						<?php if ( in_array( $bot->bot_company, array( 'OpenAI', 'Anthropic', 'Google', 'Perplexity' ) ) ) : ?>
							<span style="background: #e7f5ea; color: #00a32a; padding: 2px 6px; border-radius: 3px; font-size: 11px; margin-left: 5px;">AI</span>
						<?php endif; ?>
					</td>
					<td><?php echo esc_html( $bot->bot_company ); ?></td>
					<td><strong><?php echo number_format( $bot->count ); ?></strong></td>
					<td>
						<?php 
						$percent = $stats['bots'] > 0 ? round( ( $bot->count / $stats['bots'] ) * 100, 1 ) : 0;
						echo $percent . '%';
						?>
					</td>
				</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		<?php else : ?>
		<p style="color: #646970;">Zatím žádné přístupy od botů.</p>
		<?php endif; ?>

		<!-- Denní statistiky -->
		<h3>Přístupy po dnech</h3>
		<?php if ( ! empty( $stats['daily'] ) ) : ?>
		<table class="wp-list-table widefat fixed striped">
			<thead>
				<tr>
					<th>Datum</th>
					<th style="width: 120px;">Boti</th>
					<th style="width: 120px;">Ostatní</th>
					<th style="width: 120px;">Celkem</th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( array_slice( $stats['daily'], 0, 14 ) as $day ) : ?>
				<tr>
					<td><?php echo date_i18n( 'j. F Y', strtotime( $day->date ) ); ?></td>
					<td><?php echo number_format( $day->bots ); ?></td>
					<td><?php echo number_format( $day->humans ); ?></td>
					<td><strong><?php echo number_format( $day->bots + $day->humans ); ?></strong></td>
				</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		<?php else : ?>
		<p style="color: #646970;">Zatím žádná data.</p>
		<?php endif; ?>
		<?php
	}

	/**
	 * Tab: Log přístupů
	 */
	private function render_logs_tab() {
		$stats = $this->get_stats( 30 );
		?>
		<div style="margin: 20px 0; display: flex; justify-content: space-between; align-items: center;">
			<p style="margin: 0;">Posledních 50 přístupů k souboru llms.txt</p>
			<form method="post" style="display: inline;">
				<?php wp_nonce_field( 'okllms_clear_logs_nonce' ); ?>
				<button type="submit" name="okllms_clear_logs" class="button" onclick="return confirm('Opravdu chcete vymazat všechny logy?');">
					<span class="dashicons dashicons-trash" style="font-size: 16px; margin-top: 4px;"></span> Vymazat logy
				</button>
			</form>
		</div>

		<?php if ( ! empty( $stats['recent'] ) ) : ?>
		<table class="wp-list-table widefat fixed striped">
			<thead>
				<tr>
					<th style="width: 150px;">Čas</th>
					<th style="width: 130px;">IP adresa</th>
					<th style="width: 180px;">Bot</th>
					<th>User Agent</th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $stats['recent'] as $log ) : ?>
				<tr>
					<td><?php echo date_i18n( 'j.n.Y H:i:s', strtotime( $log->accessed_at ) ); ?></td>
					<td><code style="font-size: 12px;"><?php echo esc_html( $log->ip_address ); ?></code></td>
					<td>
						<?php if ( $log->is_bot && $log->bot_name ) : ?>
							<strong style="color: #00a32a;"><?php echo esc_html( $log->bot_name ); ?></strong>
							<br><small style="color: #646970;"><?php echo esc_html( $log->bot_company ); ?></small>
						<?php elseif ( $log->is_bot ) : ?>
							<span style="color: #dba617;">Neznámý bot</span>
						<?php else : ?>
							<span style="color: #646970;">—</span>
						<?php endif; ?>
					</td>
					<td>
						<code style="font-size: 11px; word-break: break-all; display: block; max-height: 40px; overflow: hidden;" title="<?php echo esc_attr( $log->user_agent ); ?>">
							<?php echo esc_html( substr( $log->user_agent, 0, 150 ) ); ?><?php echo strlen( $log->user_agent ) > 150 ? '...' : ''; ?>
						</code>
					</td>
				</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		<?php else : ?>
		<div style="background: #fff; padding: 40px; text-align: center; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
			<span class="dashicons dashicons-database" style="font-size: 48px; color: #ccc;"></span>
			<p style="color: #646970; margin-top: 10px;">Zatím žádné zaznamenané přístupy.</p>
			<p><a href="<?php echo esc_url( home_url( '/llms.txt' ) ); ?>" target="_blank" class="button">Otestovat llms.txt</a></p>
		</div>
		<?php endif; ?>
		<?php
	}
}

// Spuštění pluginu
new OKLLMS_Generator();
